from typing import Dict
import numpy as np

def radial_profile(arr: np.ndarray, cy: int, cx: int, r_edges: np.ndarray, reduce='mean'):
    H, W = arr.shape
    yy, xx = np.indices((H, W))
    rr = np.sqrt((yy - cy)**2 + (xx - cx)**2)
    inds = np.digitize(rr.ravel(), r_edges) - 1
    nbins = len(r_edges) - 1
    prof = np.zeros(nbins, dtype=float); counts = np.zeros(nbins, dtype=int)
    flat = arr.ravel()
    for i in range(nbins):
        m = inds == i
        if reduce == 'mean':
            prof[i] = np.mean(flat[m]) if np.any(m) else np.nan
        elif reduce == 'sum':
            prof[i] = np.sum(flat[m])
        counts[i] = int(np.sum(m))
    return prof, counts

def fit_loglog_slope(r_centers: np.ndarray, y: np.ndarray, weights=None) -> Dict[str, float]:
    m = (~np.isnan(y)) & (y > 0) & (r_centers > 0)
    if not np.any(m):
        return {"intercept": float("nan"), "slope": float("nan"), "r2": float("nan")}
    x = np.log(r_centers[m]); t = np.log(y[m])
    if weights is not None:
        w = weights[m]; w = w / (w.max() + 1e-12)
        X = np.column_stack([np.ones_like(x), x]); beta = np.linalg.pinv(np.diag(w) @ X) @ (np.diag(w) @ t)
    else:
        X = np.column_stack([np.ones_like(x), x]); beta = np.linalg.lstsq(X, t, rcond=None)[0]
    yhat = X @ beta
    ss_res = float(np.sum((t - yhat)**2)); ss_tot = float(np.sum((t - t.mean())**2) + 1e-12)
    r2 = 1.0 - ss_res/ss_tot
    return {"intercept": float(beta[0]), "slope": float(beta[1]), "r2": float(r2)}
